// ptontoa.m -- use inet_pton and inet_ntoa

#import <sys/types.h>  // for type definitions, like u_char
#import <sys/socket.h> // for AF_INET[6]
#import <netinet/in.h> // for in_addr and in6_addr
#import <arpa/inet.h>	// for inet_*
#import <errno.h>	// for errno
#import <string.h>	// for strerror
#import <stdio.h> // for printf()

/* compile with
gcc -g -Wall -o ptontoa ptontoa.m
*/

const char *in6ToChar (struct in6_addr *addr)
{
    static char s_address[INET6_ADDRSTRLEN];
    uint32_t *base;
    base = (uint32_t *)addr->s6_addr;

    snprintf (s_address, sizeof(s_address),
              "%x%x%x%x", base[0], base[1], base[2], base[3]);

    return (s_address);
    
} // in6ToChar


int main (void)
{
    struct in_addr ipv4netAddr;
    struct in6_addr ipv6netAddr;
    char ipv4Address[INET_ADDRSTRLEN];
    char ipv6Address[INET6_ADDRSTRLEN];
    int result;

    // --------------------------------------------------
    // first, IPv4 land

    // presentation to numeric
    strcpy (ipv4Address, "192.168.254.123");
    result = inet_pton (AF_INET, ipv4Address, &ipv4netAddr);
    
    if (result == 1) {
        printf ("address '%s' in binary: %x\n", 
                ipv4Address, ipv4netAddr.s_addr);
    } else if (result == 0) {
        printf ("address '%s' not parsable\n\n", ipv4Address);
    } else {
        printf ("some other error happened: %d/%s\n",
                errno, strerror(errno));
    }

    // numeric to presentation
    const char *ntopResult;
    ntopResult = inet_ntop (AF_INET, &ipv4netAddr, 
                            ipv4Address, sizeof(ipv4Address));
    
    if (ntopResult != NULL) {
        printf ("address '%x' presentation: '%s'\n", 
                ipv4netAddr.s_addr, ipv4Address);
    } else {
        printf ("some other error happened: %d/%s\n",
                errno, strerror(errno));
    }

    // --------------------------------------------------
    // now for IPv6

    strcpy (ipv6Address, "FE80:0000:0000:0000:0230:65FF:FE06:6523");
    result = inet_pton (AF_INET6, ipv6Address, &ipv6netAddr);
    
    if (result == 1) {
        printf ("address '%s'\n    in binary: %s\n", 
                ipv6Address, in6ToChar(&ipv6netAddr));
    } else if (result == 0) {
        printf ("address '%s' not parsable\n\n", ipv4Address);
    } else {
        printf ("some other error happened: %d/%s\n",
                errno, strerror(errno));
    }

    // numeric to presentation
    ntopResult = inet_ntop (AF_INET6, &ipv6netAddr, 
                            ipv6Address, sizeof(ipv6Address));
    
    if (ntopResult != NULL) {
        printf ("address '%s'\n    presentation: '%s'\n", 
                in6ToChar(&ipv6netAddr), ipv6Address);
    } else {
        printf ("some other error happened: %d/%s\n",
                errno, strerror(errno));
    }

    return (0);

} // main
