// bundleprinter-dl.m -- dynamically load some plugins and invoke functions
//                       on them, using the dlopen() family of calls

/* compile with
cc -g -o bundleprinter-dl bundleprinter-dl.m
*/

#import <dlfcn.h>	// for dlopen() and friends
#import <sys/types.h>	// for random type definition
#import <sys/dirent.h>	// for struct dirent
#import <dirent.h>	// for opendir and friends
#import <stdlib.h>	// for EXIT_SUCCESS
#import <stdio.h>	// for printf
#import <errno.h>	// for errno/strerror
#import <string.h>	// for strstr


// we need a type to coerce a void pointer to the function pointer we
// need to jump through.  Having a type makes things a bit easier
// to read

typedef int (*BNRMessageActivateFP) (void);
typedef void (*BNRMessageDeactivateFP) (void);
typedef char * (*BNRMessageMessageFP) (void);

// given a path to a plugin, load it, activate it, get the message, 
// deactivate it, and unload it

char *processPlugin (const char *path)
{
    char *message = NULL;

    void *module;
    module = dlopen (path, RTLD_LAZY);

    if (module == NULL) {
        fprintf (stderr, "couldn't load plugin at path %s.  error is %s\n",
                 path, dlerror());
        goto bailout;
    }

    BNRMessageActivateFP activator;
    BNRMessageDeactivateFP deactivator;
    BNRMessageMessageFP messagator;

    activator = dlsym (module, "BNRMessageActivate");
    deactivator = dlsym (module, "BNRMessageDeactivate");
    messagator = dlsym (module, "BNRMessageMessage");

    if (activator == NULL || deactivator == NULL
        || messagator == NULL) {
        fprintf (stderr, "could not find BNRMessage* symbol (%p %p %p)\n",
                 activator, deactivator, messagator);
        goto bailout;
    }

    int result;
    result = (activator)();
    if (!result) { // the module didn't consider itself loaded
        goto bailout;
    }

    message = (messagator)();

    (deactivator)();

  bailout:

    if (module != NULL) {
	result = (dlclose (module));
        if (result != 0) {
            fprintf (stderr, "could not dlclose %s.  Error is %s\n",
                     path, dlerror());
        }
    }
    
    return (message);
    
} // processPlugin




int main (int argc, char *argv[])
{
    DIR *directory;
    struct dirent *entry;

    // walk through the current directory

    directory = opendir (".");

    if (directory == NULL) {
	fprintf (stderr, 
		 "could not open current directory to look for plugins\n");
	fprintf (stderr, "error: %d (%s)\n", errno, strerror(errno));
	exit (EXIT_FAILURE);
    }

    while ( (entry = readdir(directory)) != NULL) {

	// if this is a file of type .msg (an extension made up for this
	// sample), process it like a plug-in

	if (strstr(entry->d_name, ".msg") != NULL) {
	    char *message;
	    message = processPlugin (entry->d_name);

	    printf ("\nmessage is: '%s'\n\n", message);
	    if (message != NULL) {
		free (message);
	    }
	}
    }

    closedir (directory);

    exit (EXIT_SUCCESS);

} // main

