// UndoDocument - a document (that doesn't save anything) to show using undo
//                in a document settingx

#import "UndoDocument.h"
#import "NoisyCount.h"

@implementation UndoDocument

// no init method since we don't initialize anything


// this is the name of the nib file that has our document window

- (NSString *) windowNibName
{
    return (@"UndoDocument");
} // windowNibName



// 'awake from nib', but for documents

- (void)windowControllerDidLoadNib: (NSWindowController *) aController
{
    [super windowControllerDidLoadNib:aController];

    // snarf any value that got entered into IB
    borkValue = [[borkString stringValue] copy];

    // just for fun, register with all the undo notifications so we can
    // see what's happening under the hood
    NSNotificationCenter *center;
    center = [NSNotificationCenter defaultCenter];

    // the notifications we're interested in
    NSArray *notifications;
    notifications = [NSArray arrayWithObjects:
			     NSUndoManagerCheckpointNotification,
			     NSUndoManagerDidOpenUndoGroupNotification,
			     NSUndoManagerDidRedoChangeNotification,
			     NSUndoManagerDidUndoChangeNotification,
			     NSUndoManagerWillCloseUndoGroupNotification,
			     NSUndoManagerWillRedoChangeNotification,
			     NSUndoManagerWillUndoChangeNotification,
			     nil];

    NSEnumerator *enumerator;
    enumerator = [notifications objectEnumerator];

    // walk the notifications and add ourselves as an observer
    
    NSString *notification;
    while ((notification = [enumerator nextObject])) {
	[center addObserver: self
		selector: @selector(logNotification:)
		name: notification
		object: nil];
    }
    
    [center addObserver: self
	    selector: @selector(textDidChangeNotification:)
	    name: NSControlTextDidChangeNotification
	    object: nil];

    // create our noisy counter, to show retains and releases in
    // various undo situatons
    noisyCount = [[NoisyCount alloc] init];
    [noisyCount retain];

} // windowControllerDidLoadNib



// clean up any mess we've made

- (void) dealloc
{
    [borkValue release];
    [noisyCount release];

    [super dealloc];

} // dealloc



// this is just a silly sample, all the document saving code has been nuked

// --------------------------------------------------



// a no-operation method that gets sent to self, and the object
// being sent is the noisyCount guy

- (void) noOp: (id) object
{
    NSLog (@"NoOp");
} // noOp



// change the value of state1, which is controlled by the "Bill"
// checkbox

- (void) setStateOne: (NSNumber *) newValue
{
    NSCellStateValue newState;
    newState = [newValue intValue];

    // if the value hasn't changed, don't do any work

    if (state1 != newState) {
	NSLog (@"setting checkbox one to %d", [newValue intValue]);
	NSUndoManager *undoManager;
	undoManager = [self undoManager];

	// to undo the change, set state1 with the current value
	[undoManager registerUndoWithTarget: self
		     selector: @selector(setStateOne:)
		     object: [NSNumber numberWithInt: state1]];

	// does not retain target, but does retain object
	// these guys aren't important to the setting of state1,
	// but show you the retains and releases going on under
	// the hood
	[undoManager registerUndoWithTarget: noisyCount
		     selector: @selector(doStuff:)
		     object: self];
	[undoManager registerUndoWithTarget: self
		     selector: @selector(noOp:)
		     object: noisyCount];

	// actually change our modelx
	state1 = newState;

	// name the action now that all undoable operations have been
	// registered
	[undoManager setActionName: [checkbox1 title]];

	// update the UI, in case we're being called due to an undo
	// or redo operation
	[checkbox1 setState: newState];
    }

} // setStateOne



// checkbox handler for the "Bill" checkbox.  It just takes the
// state, wraps it into an NSObject, and calls setStateOne: to actually
// effect the change and make the undo action

- (IBAction) handleCheckboxOne: (id) sender
{
    // make sure that any pending text operations don't confuse
    // the undo stack
    [self commitTextfield];

    NSNumber *newState;
    newState = [NSNumber numberWithInt: [sender state]];

    [self setStateOne: newState];

} // handleCheckboxOne



// --------------------------------------------------


// change the value of state2, which is controlled by the "The"
// checkbox.  This takes a scalar argument (as opposed to an NSObject-derived
// argument), so you have to use prepareWithInvocationTarget to undo
// this guy.

- (void) setStateTwo: (NSCellStateValue) newState
{
    // here's another way of doing the undo action

    if (state2 != newState) {
	NSLog (@"setting checkbox two to %d", newState);

	NSUndoManager *undoManager;
	undoManager = [self undoManager];

	// to undo the change, set state2 with the current value
	[[undoManager prepareWithInvocationTarget: self]
	    setStateTwo: state2];

	// this retains the target.
	// this isn't important to the setting of state2, but shows
	// the retains and releases being done by the toolkit
	[[undoManager prepareWithInvocationTarget: noisyCount]
	    doStuff: self];

	// actually change the model
	state2 = newState;

	// name the action now that all undoable operations have been
	// registered
	[undoManager setActionName: [checkbox2 title]];

	// update the UI, in case we're being called due to an undo
	// or redo operation
	[checkbox2 setState: newState];
    }

} // setStateTwo



// checkbox handler for the "The" checkbox.  It just takes the
// state and calls setStateTwo: with it.

- (IBAction) handleCheckboxTwo: (id) sender
{
    // make sure that any pending text operations don't confuse
    // the undo stack
    [self commitTextfield];

    NSCellStateValue newState;
    newState = [sender state];

    [self setStateTwo: newState];

} // handleCheckboxTwo



// --------------------------------------------------

// change the value of state2, which is controlled by the "Cat"
// checkbox.   Works just like setStateTwo:

- (void) setStateThree: (NSCellStateValue) newState
{
    if (state3 != newState) {
	NSLog (@"setting checkbox three to %d", newState);

	NSUndoManager *undoManager;
	undoManager = [self undoManager];

	// to undo the change, set state3 with the current value
	[[undoManager prepareWithInvocationTarget: self]
	    setStateThree: state3];

	// actually 'change' the 'model'
	state3 = newState;
	
	// name the action
	[undoManager setActionName: [checkbox3 title]];

	// update the UI
	[checkbox3 setState: newState];
    }

} // setStateThree



// checkbox handler for the "Cat" checkbox

- (IBAction) handleCheckboxThree: (id) sender
{
    // make sure that any pending text operations don't confuse
    // the undo stack
    [self commitTextfield];

    NSCellStateValue newState;
    newState = [sender state];

    [self setStateThree: newState];

} // handleCheckboxThree


// --------------------------------------------------


// change a dwarf value, based on changes from a radio group

- (void) setDwarf: (NSNumber *) newDwarfNumber
{
    int newDwarf;
    newDwarf = [newDwarfNumber intValue];

    // if the value hasn't changed, don't do any work

    if (dwarfValue != newDwarf) {

	// figure out which dwarf we're dealing with
	id cell;
	cell = [dwarfMatrix cellWithTag: newDwarf];

	NSLog (@"setting dwarf %@", [cell title]);

	NSUndoManager *undoManager;
	undoManager = [self undoManager];

	// to undo the dwarf change, remember the current dwarf
	[undoManager registerUndoWithTarget: self
		     selector: @selector(setDwarf:)
		     object: [NSNumber numberWithInt: dwarfValue]];

	dwarfValue = newDwarf;
	
	[undoManager setActionName: @"Dwarf Change"];

	[dwarfMatrix selectCellWithTag: newDwarf];
    }

} // setDwarf



// matrix handler for the user selecting a different dawrd radio buttonx
- (IBAction) handleChangeDwarf: (id) sender
{
    [self commitTextfield];

    NSLog (@"new dwarf");

    int newDwarf;
    newDwarf = [[dwarfMatrix selectedCell] tag];

    [self setDwarf: [NSNumber numberWithInt: newDwarf]];

} // handleChangeDwarf




// --------------------------------------------------


// change the sliderValue value.  Be warned, long dragging sessions
// with the slider will build up a bunch of undo operations.  See
// UndoQueue for a solution.

- (void) setSliderValue: (int) newSliderValue
{
    if (sliderValue != newSliderValue) {
	NSLog (@"setting slider value to  %d", newSliderValue);

	NSUndoManager *undoManager;
	undoManager = [self undoManager];
	
	[[undoManager prepareWithInvocationTarget: self]
	    setSliderValue: sliderValue];

	sliderValue = newSliderValue;
	
	[undoManager setActionName: @"Slider Change"];

	[slider setIntValue: newSliderValue];
    }

} // setSliderValue



// slider callback.  Turn around and tell setSliderValue to do the workx

- (IBAction) handleSlider: (id) sender
{
    [self commitTextfield];

    int newSliderValue;
    newSliderValue = [sender intValue];

    [self setSliderValue: newSliderValue];

} // handleSlider



// --------------------------------------------------

// change the borkValue, which is a string

- (void) setBorkValue: (NSString *) newBorkValue
{
    if (![borkValue isEqualTo: newBorkValue]) {
	NSLog (@"setting bork value to '%@'", 
	       newBorkValue);

	NSUndoManager *undoManager;
	undoManager = [self undoManager];

	// to undo the string change, remember the current value

	[undoManager registerUndoWithTarget: self
		     selector: @selector(setBorkValue:)
		     object: borkValue];

	// registerUndoWithTarget retains borkValue, so it's safe to
	// release it
	[borkValue release];

	// make a copy, since the string may be the mutable string
	// that the textitem is manipulating.  Isolate ourselves from
	// any future changes
	borkValue = [newBorkValue copy];

	[undoManager setActionName: @"Borkulation"];

	[borkString setStringValue: newBorkValue];
    }

} // setBorkValue



// action handler from the text field

- (IBAction) handleBorkField: (id) sender
{
    NSString *newBorkValue;
    newBorkValue = [sender stringValue];

    [self setBorkValue: newBorkValue];
    
} // handleBorkField



// some other part of the document is wanting to make an undo
// operation. Commit any changes to the text field first.

- (void) commitTextfield
{
    if (![borkValue isEqualTo: [borkString stringValue]]) {
	[self setBorkValue: [borkString stringValue]];
	NSLog (@"grouping level is %d", [[self undoManager] groupingLevel]);
	[[self undoManager] endUndoGrouping];
	[[self undoManager] beginUndoGrouping];
    }

} // commitTextfield


// --------------------------------------------------


// purge the undo stack, to start back with a clean slate

- (IBAction) clearUndo: (id) sender
{
    NSLog (@"purging undo stack");

    [[self undoManager] removeAllActions];
    [self updateChangeCount: NSChangeCleared];

} // clearUndo



// open and close an undo group, which will collect a bunch of
// independent undo operations under one umbrella.

- (IBAction) handleUndoGroup: (id) sender
{
    if ([sender state] == NSOnState) {
	NSLog (@"beginning new undo group");
	// begin a new group
	[[self undoManager] beginUndoGrouping];
    } else {
	NSLog (@"ending new undo group");
	// end the group
	[[self undoManager] setActionName: [groupName stringValue]];
	// doesn't quite work
	[[self undoManager] endUndoGrouping];
    }

} // handleUndoGroup



// --------------------------------------------------

// show all undo-related notifications that float by

- (void) logNotification: (NSNotification *) notification
{
    NSLog (@"%@", [notification  name]);

} // logNotification



@end
